<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Access\Wordpress;

use YOOtheme\Config;
use ZOOlanders\YOOessentials\Access\AccessRule;

class UserAccessRule implements AccessRule
{
    /**
     * @var Config
     */
    private $config;

    public function __construct(Config $config)
    {
        $this->config = $config;
    }

    public function group() : string
    {
        return 'user';
    }

    public function name() : string
    {
        return 'User Access';
    }

    public function namespace() : string
    {
        return 'yooessentials_access_useraccess';
    }

    public function description() : string
    {
        return 'Validates if the user Access Roles matches the selection.';
    }

    public function resolve($props, $node) : bool
    {
        if (!isset($props->levels)) {
            return true;
        }

        $user = $this->config->get('user');
        $roles = $user->roles;

        if ($props->guest ?? false) {
            return !$user->exists();
        }

        $selection = $props->levels ?? [];
        $strict = $props->strict ?? false;

        // legacy code, prior v1.2 guest was an option and not an injected role
        if ($props->guest ?? false) {
            $selection[] = '_guest';
        }

        $missingRoles = array_diff($selection, $roles);

        return $strict
            ? count($missingRoles) === 0
            : count($missingRoles) < count($selection);
    }

    public function fields() : array
    {
        return [
            "{$this->namespace()}.levels" => [
                'type' => 'select',
                'description' => 'The access roles that the current user must met. Use the shift or ctrl/cmd key to select multiple entries.',
                'attrs' => [
                    'multiple' => true,
                    'class' => 'uk-height-small uk-resize-vertical'
                ],
                'options' => $this->getUserRoles(),
                'enable' => "!{$this->namespace()}.guest"
            ],
            "{$this->namespace()}.strict" => [
                'text' => 'All selected roles must be met',
                'type' => 'checkbox',
                'enable' => "!{$this->namespace()}.guest"
            ],
            "{$this->namespace()}.guest" => [
                'text' => 'Validate if user is Guest',
                'type' => 'checkbox'
            ]
        ];
    }

    protected function getUserRoles() : array
    {
        static $list = [];

        if (empty($list)) {
            require_once \ABSPATH . 'wp-admin/includes/user.php';

            $roles = array_reverse(\get_editable_roles());

            $list['Guest'] = '_guest';

            foreach ($roles as $id => $role) {
                $list[$role['name']] = $id;
            }
        }

        return $list;
    }
}
