<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Access\Rule;

use DateTime;
use DateTimeZone;
use YOOtheme\Config;
use ZOOlanders\YOOessentials\Access\AccessRule;

class TimeRule implements AccessRule
{
    /**
     * @var DateTimeZone
     */
    protected $tz;

    /**
     * @var String
     */
    protected $format = 'H:i';

    public function __construct(Config $config)
    {
        $this->tz = $config->get('yooessentials.timezone');
    }

    public function group() : string
    {
        return 'datetime';
    }

    public function name() : string
    {
        return 'Time';
    }

    public function namespace() : string
    {
        return 'yooessentials_access_time';
    }

    public function description() : string
    {
        return 'Validates if the current time is in the specified range.';
    }

    public function resolve($props, $node) : bool
    {
        $now = $this->now();
        $publishUp = trim($props->publish_up ?? '');
        $publishDown = trim($props->publish_down ?? '');

        try {
            $publishUp = DateTime::createFromFormat($this->format, $publishUp, $this->tz);
            $publishDown = DateTime::createFromFormat($this->format, $publishDown, $this->tz);
        } catch (\Exception $e) {
            return true;
        }

        if ($publishUp and (int) $publishUp->format('U') > (int) $now->format('U')) {
            return false;
        }

        if ($publishDown and (int) $publishDown->format('U') < (int) $now->format('U')) {
            return false;
        }

        return true;
    }

    public function now() : DateTime
    {
        return new DateTime('now', $this->tz);
    }

    public function fields() : array
    {
        return [
            '_grid' => [
                'type' => 'grid',
                'width' => '1-2',
                'description' => 'Timezone from Site configuration is automatically applied.',
                'fields' => [
                    "{$this->namespace()}.publish_up" => [
                        'label' => 'From',
                        'attrs' => [
                            'type' => 'time',
                            'placeholder' => 'H:i'
                        ]
                    ],
                    "{$this->namespace()}.publish_down" => [
                        'label' => 'To',
                        'attrs' => [
                            'type' => 'time',
                            'placeholder' => 'H:i'
                        ]
                    ]
                ]
            ]
        ];
    }
}
