<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Form;

use YOOtheme\Application;
use YOOtheme\Arr;
use YOOtheme\Config;
use YOOtheme\Metadata;
use YOOtheme\Middleware;
use YOOtheme\Path;
use ZOOlanders\YOOessentials\Addon;
use ZOOlanders\YOOessentials\Form\Action\Action;
use ZOOlanders\YOOessentials\Form\Http\FormSubmissionResponse;

class FormListener
{
    /**
     * @param FormSubmissionResponse $response
     * @param callable $next
     */
    public static function handleFormActions(Application $app, $response, $next)
    {
        $actions = $response->submission()->form()->actions();
        $listeners = [];
        foreach ($actions as $actionType => $actionConfigs) {
            $listeners = array_merge($listeners, array_map(function (array $actionConfig) use ($app, $actionType) {
                /** @var Action $actionClass */
                $actionClass = $app($actionType);

                return $actionClass->setConfig($actionConfig);
            }, $actionConfigs));
        }

        $middleware = new Middleware(function () use ($response) {
            return $response;
        }, $listeners);

        /** @var FormSubmissionResponse $response */
        return $middleware($response);
    }

    public static function initCustomizer(Config $config, Metadata $metadata, FormService $formService)
    {
        $config->addFile('customizer', Path::get('../config/customizer.json'));

        if (!Addon::enabled(Form::class)) {
            return;
        }

        // set config
        $config->addFile('customizer.panels.yooessentials-form-config', Path::get('../config/form-config.php'));

        $metadata->set('script:yooessentials-customizer-form', ['src' => '~yooessentials_url/modules/form/assets/customizer.min.js', 'defer' => true]);
        $metadata->set('style:yooessentials-customizer-form', ['href' => '~yooessentials_url/modules/form/assets/customizer.min.css', 'defer' => true]);
    }

    public static function addFormPanel(Config $config, $type)
    {
        if (!Addon::enabled(Form::class)) {
            return $type;
        }

        // constraint types
        if (!in_array($type['name'], ['section', 'column'])) {
            return $type;
        }

        // make sure the main fieldset is set
        if (!Arr::has($type, 'fieldset.default')) {
            return $type;
        }

        $tabs = array_reduce($type['fieldset']['default']['fields'], function ($carry, $v) {
            return array_merge($carry, [$v['title'] ?? null]);
        }, []);

        if (($index = array_search('Advanced', $tabs)) === false) {
            return $type;
        }

        $statusField = [
            'type' => 'checkbox',
            'name' => 'yooessentials_form.state',
            'label' => 'Form',
            'text' => 'Enable as Form Area'
        ];

        $configButton = [
            'name' => '_yooessentials_form',
            'text' => 'Configuration',
            'type' => 'button-panel',
            'panel' => 'yooessentials-form-config',
            'enable' => 'yooessentials_form.state',
            'description' => 'Enable this element as a Form Area, and set it submission configuration.'
        ];

        // set button right after status field
        Arr::splice($type['fieldset']['default']['fields'][$index]['fields'], 2, 0, [$statusField, $configButton]);

        return $type;
    }
}
