<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Form;

use function YOOtheme\app;
use YOOtheme\Arr;
use YOOtheme\Builder;
use YOOtheme\Config;
use ZOOlanders\YOOessentials\Addon;
use ZOOlanders\YOOessentials\Form\Http\FormSubmissionRequest;
use ZOOlanders\YOOessentials\Form\Http\FormSubmissionResponse;
use ZOOlanders\YOOessentials\Util;
use ZOOlanders\YOOessentials\Vendor\Respect\Validation\Exceptions\ValidationException;
use ZOOlanders\YOOessentials\Vendor\Respect\Validation\Validator;

class FormService
{
    /** @var bool */
    protected $enabled = true;

    /**
     * @var array
     */
    public $actions = [];

    /**
     * @var callable
     */
    protected $loader;

    /**
     * @var FormConfigCache
     */
    protected $cache;

    /**
     * @var Builder
     */
    protected $builder;

    public function __construct(Config $config, Builder $builder)
    {
        $this->loader = [
            $config,
            'loadFile'
        ];

        $this->cache = new FormConfigCache('form');
        $this->builder = $builder;

        if (!Addon::enabled(Form::class)) {
            return;
        }

        $builder->addTransform('render', new FormTransform($this, $builder));
    }

    public function loadForm(string $id): Form
    {
        $config = $this->loadConfig($id);

        return new Form($id, $config);
    }

    /**
     * @param  string  $id
     * @return object|void|null
     */
    public function loadFormNode(string $id)
    {
        return $this->builder->load(json_encode([
            'id' => $id,
            'type' => 'yooessentials_form'
        ]), ['context' => 'render']);
    }

    /**
     * Load the Config for a form
     * @param  string  $formId
     * @return array
     */
    public function loadConfig(string $formId): array
    {
        $cached = $this->cache->resolve(sprintf('%s.php', $formId));

        if (is_file($cached)) {
            return include $cached;
        }

        return [];
    }

    /**
     * Save config for a form
     * @param  string  $id
     * @param  array  $config
     */
    public function saveConfig(string $id, array $config): void
    {
        $cached = sprintf('%s.php', $id);

        $value = Util::compileValue(json_decode(json_encode($config), true));

        $this->cache->set($cached, "<?php\n\nreturn {$value};\n");
    }

    /**
     * Add action class.
     *
     * @return $this
     */
    public function addAction($action)
    {
        if (!isset($this->actions[$action])) {
            $this->actions[$action] = app($action);
        }

        return $this;
    }

    public function actions(): array
    {
        return $this->actions;
    }

    public function validateElements(FormSubmissionRequest $submission): array
    {
        $errors = [];

        foreach ($submission->config()['controls'] ?? [] as $control) {
            $type = $this->builder->types[$control['type']] ?? false;

            if (!$type) {
                continue;
            }

            $validation = $type->data['validation'] ?? null;
            if (!is_callable($validation)) {
                continue;
            }

            $name = $control['name'];
            $props = $control['props'];
            $label = $props['label'] ?? $name;
            $value = $submission->data($name);
            $control['value'] = $value;

            $validator = new Validator();
            $validator->setName($label);

            try {
                $validation($control, $validator, $submission);
                $validator->check($value);
            } catch (ValidationException $exception) {
                if ($message = $props['error_message'] ?? false) {
                    $message = str_replace('{fieldlabel}', $label ?: $name, $message);
                    $exception->updateTemplate($message);
                }

                $errors[$name] = Arr::wrap($exception->getMessage());
            }
        }

        return $errors;
    }

    public function processElementSubmission(FormSubmissionRequest $submission, FormSubmissionResponse $submissionResponse): void
    {
        foreach ($submission->form()->config()['controls'] ?? [] as $control) {
            $type = $this->builder->types[$control['type']] ?? false;

            if ($type && is_callable($type->data['submission'] ?? null)) {
                $type->data['submission']($control, $submission, $submissionResponse);
            }
        }
    }
}
