<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Icons;

use YOOtheme\Config;
use YOOtheme\File;
use YOOtheme\Path;
use YOOtheme\Str;
use ZOOlanders\YOOessentials\Vendor\Symfony\Contracts\Cache\CacheInterface;
use ZOOlanders\YOOessentials\Vendor\Symfony\Contracts\Cache\ItemInterface;

class Icons
{
    /**
     * @var CacheInterface
     */
    public $cache;

    /**
     * @var Config
     */
    public $config;

    /**
     * @var array
     */
    public $collectionsUbications = [
        '~yooessentials/modules/icons/icons'
    ];

    /**
     * Queue for icons to be loaded
     *
     * @var array
     */
    public $queue = [];

    /**
     * Constructor.
     */
    public function __construct(Config $config, CacheInterface $cache)
    {
        $this->config = $config;
        $this->cache = $cache;
    }

    public function load(string $icon): ?string
    {
        $isProvided = Str::contains($icon, '--');

        if (!$isProvided) {
            return null;
        }

        $iconKey = sha1("icon-{$icon}.svg");
        $content = $this->cache->get($iconKey, function (ItemInterface $item) use ($icon) {
            if ($file = $this->find($icon)) {
                return File::getContents($file);
            }

            return null;
        });

        if ($content) {
            $this->queue[$icon] = $content;
        }

        return $content;
    }

    /**
     * Returns the icons that were queued for loading, and their content
     *
     * @return array
     */
    public function getQueued(): array
    {
        return $this->queue;
    }

    /**
     * Add a directory where to look for icons collections
     *
     * @param string $dir
     * @return $this
     */
    public function addCollectionsDir(string $dir): self
    {
        $this->collectionsUbications[] = $dir;

        return $this;
    }

    /**
     * Finds the icon and returns it first ubication path
     *
     * @param string $icon
     * @return string
     */
    protected function find(string $icon): ?string
    {
        list('collection' => $collection, 'group' => $group, 'name' => $name) = $this->parseValue($icon);

        $basepath = Path::join($group, $name);

        if ($collection === 'myicons' and $dir = $this->config->get('theme.childDir') ?? false and $file = File::find("$dir/myicons/$basepath.svg")) {
            return $file;
        }

        foreach ($this->collectionsUbications as $dir) {
            if ($file = File::find("$dir/$collection/$basepath.svg")) {
                return $file;
            }
        }

        return null;
    }

    /**
     * Parse icon info from it value
     *
     * @param string $icon
     * @return array
     */
    protected function parseValue(string $icon): array
    {
        preg_match('/^([^-]*)-?(.*)--(.*)$/', $icon, $matches);
        list($match, $collection, $group, $name) = $matches;

        return compact('collection', 'group', 'name');
    }

    /**
     * Retrieve icons from a node
     *
     * @return Array
     */
    public function retrieveIcons($node, $type)
    {
        return array_unique(array_merge(
            $this->_retrieveIconsFields($node, $type),
            $this->_retrieveIconsHtml($node, $type)
        ));
    }

    /**
     * Retrieve icons from known icon fields
     *
     * @return Array
     */
    public function _retrieveIconsFields($node, $type)
    {
        $icons = [];
        $fields = $type->data['fields'] ?? [];

        // get all fields key of type 'icon'
        $fieldsKeys = array_keys(array_filter($fields, function ($v) {
            return in_array($v['type'] ?? '', [
                'icon',
                'button_icon'
            ]);
        }));

        foreach ($fieldsKeys as $key) {
            $value = $node->props[$key] ?? false;

            if (is_string($value)) {
                // icon could be set raw or with attributes (unusual but still expected),
                // we always explode and assume the icon name is set first
                list($iconName) = explode(';', $value);

                $icons[] = trim($iconName);
            }
        }

        return $icons;
    }

    /**
     * Retrieve icons from HTML declarations, eg <span uk-icon/>
     *
     * @return array
     */
    protected function _retrieveIconsHtml($node, $type)
    {
        $icons = [];
        $content = [];

        // iterate fields content
        foreach (array_keys($type->data['fields'] ?? []) as $key) {
            if ($value = $node->props[$key] ?? false and is_string($value)) {
                $content[] = $value;
            }
        }

        // iterate source content filters (those can store html)
        foreach ($node->source->props->source->filters ?? [] as $value) {
            if (is_string($value)) {
                $content[] = $value;
            }
        }

        // match all icons set as html
        foreach ($content as $value) {
            if (strpos($value, 'uk-icon') !== false) {
                if (preg_match_all('/<[^\/]*?uk-icon.*?>/', $value, $allMatches)) {
                    foreach ($allMatches[0] as $match) {
                        if (preg_match('/[\w-]*--[\w-]*/', $match, $matches)) {
                            $icons[] = $matches[0];
                        }
                    }
                }
            }
        }

        return $icons;
    }
}
