<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Icons;

use YOOtheme\Arr;
use YOOtheme\Config;
use YOOtheme\File;
use YOOtheme\Http\Request;
use YOOtheme\Http\Response;
use YOOtheme\Path;

class IconsController
{
    /**
     * @var Icons
     */
    public $icons;

    /**
     * @var Config
     */
    public $config;

    /**
     * @var array
     */
    public $ubications;

    /**
     * Constructor.
     */
    public function __construct(Icons $icons, Config $config)
    {
        $this->icons = $icons;
        $this->config = $config;

        $this->ubications = array_map(function ($ubicaction) {
            return Path::resolveAlias($ubicaction);
        }, $this->icons->collectionsUbications);
    }

    public function getIcons(Request $request, Response $response)
    {
        try {
            $offset = (int) $request->getParam('offset', 0);
            $length = (int) $request->getParam('length', 200);
            $search = $request->getParam('search');
            $group = $request->getParam('group');
            $collection = $request->getParam('collection');

            $ubications = $collection === 'myicons'
                ? [$this->config->get('theme.childDir')]
                : $this->ubications;

            $result = $this->fetchIcons($ubications, $offset, $length, $search, $collection, $group);

            return $response->withJson($result);
        } catch (\Exception $e) {
            $request->abort(400, $e->getMessage());
        }
    }

    public function getCollections(Request $request, Response $response)
    {
        try {
            $result = array_map(function ($collection) {
                return [
                    'name' => $collection->name(),
                    'title' => $collection->title(),
                    'groups' => $collection->groups()
                ];
            }, array_values($this->fetchCollections()));

            return $response->withJson($result);
        } catch (\Exception $e) {
            $request->abort(400, $e->getMessage());
        }
    }

    protected function fetchCollections()
    {
        static $collections = [];

        if (empty($collections)) {
            if ($dir = $this->config->get('theme.childDir')) {
                $collections['myicons'] = new Collection("$dir/myicons", [
                    'name' => 'myicons',
                    'title' => 'My Icons'
                ]);
            }

            foreach ($this->ubications as $dir) {
                $paths = Arr::filter(File::glob("{$dir}/*"), [File::class, 'isDir']);

                foreach ($paths as $path) {
                    $name = basename($path);

                    if (!isset($collections[$name])) {
                        $collections[$name] = new Collection($path);
                    }
                }
            }
        }

        return array_values($collections);
    }

    protected function fetchIcons($ubications, $offset, $length, $search = null, $collection = null, $group = null)
    {
        $collections = array_map(function ($collection) {
            return $collection->name();
        }, $this->fetchCollections());

        $namePattern = $search ? "*$search*" : '*';
        $collectionPattern = $collection ? "/$collection/" : '/*/';
        $groupPattern = $group === '__main' ? '' : ($group ? "$group/" : '{,*/}');

        $pattern = '{' . implode(',', $ubications) . '}' . $collectionPattern . $groupPattern . "$namePattern.svg";

        $icons = File::glob($pattern);
        $total = count($icons);
        $icons = array_splice($icons, $offset, $length);

        $data = new \stdClass;

        foreach ($icons as $path) {
            $group = basename(dirname($path));
            $collection = basename(dirname(dirname($path)));

            if (!in_array($collection, $collections)) {
                $collection = $group;
                $group = null;
            }

            $icon = basename($path, '.svg');
            $content = File::getContents($path);

            $key = $group
                ? sprintf('%s-%s--%s', $collection, $group, $icon)
                : sprintf('%s--%s', $collection, $icon);

            $data->$key = $content;
        }

        return compact('data', 'total', 'offset', 'length', 'search', 'collection', 'group');
    }
}
