<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Source;

use DateTime;

class ConnectedAccount
{
    public const EXPIRE_THRESHOLD = 10 * 86400;

    /** @var string */
    public $driver;

    /** @var string */
    public $id;

    /** @var string */
    public $accessToken;

    /** @var array */
    public $scopes = [];

    /** @var string|null */
    public $fullName;

    /** @var string|null */
    public $username;

    /** @var string|null */
    public $email;

    /** @var string|null */
    public $refreshToken;

    /** @var int|null */
    public $expiresIn;

    /** @var DateTime|null */
    public $expiresAt = null;

    /** @var string|null */
    public $avatar;

    public function __construct(array $config = [])
    {
        $expiresAt = null;
        $expiresIn = $config['expiresIn'] ?? null;

        if ($expiresIn !== null) {
            $now = (int) (new DateTime())->format('U');
            $expiresAt = new DateTime();
            $expiresAt->setTimestamp($now + (int) $expiresIn);
        }

        if (($config['expiresAt'] ?? null)) {
            $expiresAt = new DateTime();
            if (is_array($config['expiresAt'])) {
                $expiresAt = new DateTime($config['expiresAt']['date'] ?? null);
            } else {
                $expiresAt->setTimestamp($config['expiresAt']);
            }
        }

        $this->driver = $config['driver'] ?? '';
        $this->id = (string) ($config['id'] ?? '');
        $this->accessToken = $config['accessToken'] ?? '';
        $this->scopes = $config['scopes'] ?? [];
        $this->fullName = $config['fullName'] ?? null;
        $this->username = $config['username'] ?? null;
        $this->email = $config['email'] ?? null;
        $this->refreshToken = $config['refreshToken'] ?? null;
        $this->expiresIn = $expiresIn;
        $this->expiresAt = $expiresAt;
        $this->avatar = $config['avatar'] ?? null;
    }

    public function isTokenExpiring(): bool
    {
        $expiresIn = $this->expiresIn();
        if (!$expiresIn) {
            return false;
        }

        if ($expiresIn < self::EXPIRE_THRESHOLD) {
            return false;
        }

        return true;
    }

    public function expiresIn(): ?int
    {
        if (!$this->expiresAt) {
            return $this->expiresIn;
        }

        $now = (int) (new DateTime())->format('U');
        $expiration = (int) ($this->expiresAt->format('U'));

        return $expiration - $now;
    }

    public function toArray(): array
    {
        return [
            'driver' => $this->driver,
            'id' => $this->id,
            'accessToken' => $this->accessToken,
            'scopes' => $this->scopes,
            'fullName' => $this->fullName,
            'username' => $this->username,
            'email' => $this->email,
            'refreshToken' => $this->refreshToken,
            'expiresIn' => $this->expiresIn,
            'expiresAt' => $this->expiresAt ? $this->expiresAt->format('U') : null,
            'avatar' => $this->avatar,
        ];
    }
}
