<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Source\Providers\GoogleSheet;

use function YOOtheme\app;
use YOOtheme\Builder\Source;
use YOOtheme\Event;
use YOOtheme\Path;
use YOOtheme\Str;
use ZOOlanders\YOOessentials\Source\ConnectedAccount;
use ZOOlanders\YOOessentials\Source\ConnectedAccountsService;
use ZOOlanders\YOOessentials\Source\SourceProviderInterface;
use ZOOlanders\YOOessentials\Source\SourceService;
use ZOOlanders\YOOessentials\Vendor\Google_Client;
use ZOOlanders\YOOessentials\Vendor\Google_Service_Sheets;
use ZOOlanders\YOOessentials\Vendor\Google_Service_Sheets_Spreadsheet;

class Provider implements SourceProviderInterface
{
    use HasGoogleClient;

    public const SHEET_START_COLUMN = 'A';
    public const SHEET_END_COLUMN = 'Z';

    public function config(): object
    {
        return (object) app()->config->loadFile(Path::get('./provider.json'));
    }

    public function initSource(Source $source, array $config = []): void
    {
        foreach ($config ?? [] as $args) {
            $sheetId = $args['sheet_id'] ?? '';

            try {
                $accountId = self::googleAccount($sheetId);
                if (!$accountId) {
                    continue;
                }

                $client = self::googleClient($accountId);
                if (!$client) {
                    continue;
                }

                $spreadSheet = self::googleSpreadSheet($client, $sheetId);
                if (!$spreadSheet) {
                    continue;
                }

                $name = self::getQueryName($args);
                $label = self::getTypeLabel($args);
                $type = self::getTypeName(self::getHeaders($args, $spreadSheet, $client));

                $source->queryType(Type\SheetQueryType::config($name, $label, $type, $sheetId));
                $source->objectType($type, Type\SheetType::config($sheetId, $client));
            } catch (\Exception $e) {
                Event::emit('yooessentials.error', [
                    'addon' => 'source',
                    'provider' => 'google-sheet',
                    'error' => $e->getMessage()
                ]);
            }
        }
    }

    public static function getQueryName(array $args): string
    {
        $id = self::getQueryId($args);

        return "googleSheet_{$id}_query";
    }

    public static function getQueryId(array $args): string
    {
        $createdOn = $args['_meta']['created_on'] ?? '';

        // legacy
        if (version_compare($createdOn, '1.2.4', '<=')) {
            $sheetId = $args['sheet_id'] ?? '';

            return Str::camelCase($sheetId);
        }

        return $args['id'];
    }

    public static function getTypeName(array $headers): string
    {
        return 'googleSheet_' . sha1(json_encode(array_filter($headers)));
    }

    public static function getTypeLabel(array $args): string
    {
        return $args['name'] ?? "Sheet - {$args['id']}";
    }

    public static function getHeaders(array $args, Google_Service_Sheets_Spreadsheet $spreadSheet, Google_Client $client): array
    {
        /** @var Google_Service_Sheets $service */
        $service = new Google_Service_Sheets($client);

        $start = $args['start_column'] ?? self::SHEET_START_COLUMN;
        $end = $args['end_column'] ?? self::SHEET_END_COLUMN;
        $headerInterval = "{$start}1:{$end}1";

        $headers = $service->spreadsheets_values->get($spreadSheet->spreadsheetId, $headerInterval)->getValues();
        $headers = $headers ? array_shift($headers) : [];

        return $headers;
    }

    public static function getValues(array $args, Google_Service_Sheets_Spreadsheet $spreadSheet, Google_Client $client): array
    {
        /** @var Google_Service_Sheets $service */
        $service = new Google_Service_Sheets($client);

        $offset = $args['offset'] ?? 0;
        $limit = $args['limit'] ?? 20;

        if ($offset < 0) {
            $offset = 0;
        }

        // force a limit
        if ($limit <= 0) {
            $limit = 1000;
        }

        // skip the header, and sheets starts from 1
        $offset += 2;
        $limit += ($offset - 1);

        $start = $args['start_column'] ?? self::SHEET_START_COLUMN;
        $end = $args['end_column'] ?? self::SHEET_END_COLUMN;
        $interval = "{$start}{$offset}:{$end}{$limit}";

        return $service->spreadsheets_values->get($spreadSheet->spreadsheetId, $interval)->getValues();
    }

    public static function googleAccount(string $sheetId): ?string
    {
        $configs = app(SourceService::class)->sources('google_sheet');

        foreach ($configs as $row) {
            if (($row['sheet_id'] ?? '') === $sheetId) {
                return $row['user_id'] ?? null;
            }
        }

        return null;
    }

    public static function googleClient(string $accountId): ?Google_Client
    {
        /** @var ConnectedAccountsService $service */
        $service = app(ConnectedAccountsService::class);
        $accounts = $service->accounts('google');

        /** @var ConnectedAccount $account */
        foreach ($accounts as $account) {
            if ($account->id === $accountId) {
                return self::createGoogleClient($account);
            }
        }

        return null;
    }

    public static function googleSpreadSheet(Google_Client $client, string $sheetId, bool $tryRefreshing = true): ?Google_Service_Sheets_Spreadsheet
    {
        try {
            $service = new Google_Service_Sheets($client);

            return $service->spreadsheets->get($sheetId);
        } catch (\Google\Service\Exception $e) {
            // Try refreshing the token but avoid infinite loop
            if ($e->getCode() === 401 && $tryRefreshing) {
                $client->fetchAccessTokenWithRefreshToken();

                return self::googleSpreadSheet($client, $sheetId, false);
            }
        }

        return null;
    }
}
