<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Source\Providers\Instagram\Type;

use function YOOtheme\app;
use YOOtheme\Event;
use ZOOlanders\YOOessentials\Source\ConnectedAccount;
use ZOOlanders\YOOessentials\Source\Providers\Instagram\InstagramApi;
use ZOOlanders\YOOessentials\Vendor\Symfony\Component\Cache\Adapter\FilesystemAdapter;
use ZOOlanders\YOOessentials\Vendor\Symfony\Contracts\Cache\CacheInterface;
use ZOOlanders\YOOessentials\Vendor\Symfony\Contracts\Cache\ItemInterface;

class InstagramMediaQueryType
{
    /**
     * @return array
     */
    public static function config(string $name, string $label, string $userId, ?string $pageId = null)
    {
        return [

            'fields' => [

                $name => [

                    'type' => [
                        'listOf' => 'InstagramMedia',
                    ],

                    'args' => [
                        'user_id' => [
                            'type' => 'ID',
                        ],
                        'page_id' => [
                            'type' => 'ID',
                        ],
                        'limit' => [
                            'type' => 'Int',
                        ],
                        'cache' => [
                            'type' => 'Int'
                        ],
                        'media_type' => [
                            'type' => 'String'
                        ],
                    ],

                    'metadata' => [
                        'group' => 'Instagram',
                        'label' => $label,
                        'fields' => [
                            'limit' => [
                                'label' => 'Limit',
                                'description' => 'Set the limit to specify the total of medias to fetch.',
                                'type' => 'number',
                                'default' => 20,
                            ],
                            'cache' => [
                                'type' => 'number',
                                'label' => 'Cache Time',
                                'description' => 'Set the Cache duration in seconds.',
                                'default' => 3600,
                            ],
                            'media_type' => [
                                'type' => 'select',
                                'label' => 'Media Types',
                                'description' => 'Choose which types of media to show',
                                'default' => 'all',
                                'options' => [
                                    'All' => 'all',
                                    'Only Images' => 'images',
                                    'Only Video' => 'videos'
                                ]
                            ],
                        ]
                    ],

                    'extensions' => [
                        'call' => [
                            'func' => __CLASS__ . '::resolve',
                            'args' => [
                                'user_id' => $userId,
                                'page_id' => $pageId,
                            ]
                        ]
                    ],

                ],

            ],

        ];
    }

    public static function resolve($root, array $args)
    {
        $userId = ($args['user_id'] ?? null);
        $pageId = ($args['page_id'] ?? null);

        if (!$userId && !$pageId) {
            return [];
        }

        /** @var FilesystemAdapter $cache */
        $cache = app(CacheInterface::class);
        $cacheKey = 'instagram-' . sha1(json_encode($args));

        $medias = $cache->get($cacheKey, function (ItemInterface $item) use ($args) {
            $item->expiresAfter($args['cache'] ?? 3600);

            /** @var InstagramApi $instagram */
            $instagram = app(InstagramApi::class);
            if (!$instagram->isConfigured()) {
                Event::emit('yooessentials.error', [
                    'addon' => 'source',
                    'provider' => 'instagram',
                    'error' => 'Instagram Not Configured'
                ]);

                return [];
            }

            $result = self::getMedias($instagram, $args);

            return $result;
        });

        // null cache if invalud medias
        if (!$medias || empty($medias)) {
            $cache->delete($cacheKey);
        }

        return $medias;
    }

    protected static function getMedias(InstagramApi $instagram, array $args = []): array
    {
        $limit = $args['limit'] ?? 20;

        $userId = $args['user_id'] ?? null;
        $pageId = $args['page_id'] ?? null;

        /** @var ConnectedAccount $config */
        $config = $instagram->configForUser($userId);

        if ($pageId && ($config->driver !== 'instagrambasic')) {
            return self::parseMedia($instagram->mediaForPage($pageId, $userId, $limit), $args);
        }

        return self::parseMedia($instagram->mediaForUser($userId, $limit), $args);
    }

    protected static function parseMedia(array $medias, array $filters = []): array
    {
        return array_filter(array_map(function ($media) use ($filters) {
            $media['tags'] = [];

            if (!isset($media['caption'])) {
                return $media;
            }

            preg_match_all('/(#\w+)/', $media['caption'] ?? '', $matches);
            $tags = $matches[0] ?: [];

            $media['tags'] = $tags;

            $mediaType = $media['media_type'] ?? 'IMAGE';
            $filterMedia = $filters['media_type'] ?? 'all';

            switch ($filterMedia) {
                case 'videos':
                    if ($mediaType !== 'VIDEO') {
                        return null;
                    }

                    break;
                case 'images':
                    if ($mediaType !== 'IMAGE') {
                        return null;
                    }

                    break;
                case 'all':
                default:
                    break;
            }

            return $media;
        }, $medias));
    }
}
